/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.netcdf;

import gov.nasa.giss.netcdf.NcAxisException;
import gov.nasa.giss.netcdf.NcAxisType;
import gov.nasa.giss.netcdf.NcDataset;
import gov.nasa.giss.netcdf.NcVarUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.ma2.Array;
import ucar.nc2.Attribute;
import ucar.nc2.constants.AxisType;
import ucar.nc2.dataset.CoordinateAxis;
import ucar.nc2.dataset.CoordinateAxis2D;
import ucar.nc2.dataset.VariableDS;
import ucar.units.Unit;
import ucar.units.UnitFormat;
import ucar.units.UnitFormatManager;

public class NcAxis {
    private static Logger logger_ = LoggerFactory.getLogger(NcAxis.class);
    private String name_;
    private CoordinateAxis axis_;
    private AxisType type_;
    private NcDataset dataset_;
    private int numVals_;
    private double[] values_;
    private double[][] bounds_;
    private boolean flipped_;
    private boolean wraparound_;
    private int redundant_;
    private static UnitFormat unitFormat_;

    public NcAxis(NcAxisType type, String name, int length) {
        switch (type) {
            case GEOX: {
                this.type_ = AxisType.GeoX;
                break;
            }
            case GEOY: {
                this.type_ = AxisType.GeoY;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to create axis of type " + (Object)((Object)type));
            }
        }
        this.name_ = name;
        this.numVals_ = length;
        this.values_ = new double[this.numVals_];
        this.bounds_ = new double[this.numVals_][2];
        for (int i = 0; i < this.numVals_; ++i) {
            double val;
            this.values_[i] = val = (double)i;
            this.bounds_[i][0] = val - 0.5;
            this.bounds_[i][1] = val + 0.5;
        }
    }

    public NcAxis(NcAxisType type, String name, double[] values) {
        switch (type) {
            case LONGITUDE: {
                this.type_ = AxisType.Lon;
                break;
            }
            case LATITUDE: {
                this.type_ = AxisType.Lat;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unable to create axis of type " + (Object)((Object)type));
            }
        }
        this.name_ = name;
        this.numVals_ = values.length;
        int last = this.numVals_ - 1;
        this.values_ = new double[this.numVals_];
        this.bounds_ = new double[this.numVals_][2];
        for (int i = 0; i < this.numVals_; ++i) {
            this.values_[i] = values[i];
            if (i == 0) {
                this.bounds_[0][0] = values[0];
                this.bounds_[i][1] = 0.5 * (values[0] + values[1]);
                continue;
            }
            if (i == last) {
                this.bounds_[i][0] = 0.5 * (values[last - 1] + values[last]);
                this.bounds_[i][1] = values[last];
                continue;
            }
            this.bounds_[i][0] = 0.5 * (values[i - 1] + values[i]);
            this.bounds_[i][1] = 0.5 * (values[i] + values[i + 1]);
        }
        if (this.type_ == AxisType.Lon) {
            double gap1 = values[1] - values[0];
            double gapX = values[last] - values[last - 1];
            this.bounds_[0][0] = values[0] - gap1;
            this.bounds_[last][1] = values[last] + gapX;
        }
    }

    public NcAxis(NcDataset dataset, CoordinateAxis axis) {
        if (dataset == null) {
            throw new IllegalArgumentException("Null dataset argument.");
        }
        if (axis == null) {
            throw new IllegalArgumentException("Null axis argument.");
        }
        if (axis instanceof CoordinateAxis2D) {
            throw new IllegalArgumentException("CoordinateAxis2D found where 1D expected.");
        }
        this.dataset_ = dataset;
        this.axis_ = axis;
        this.type_ = this.axis_.getAxisType();
        this.name_ = this.axis_.getFullName();
        this.numVals_ = this.axis_.getShape()[0];
        this.values_ = new double[this.numVals_];
        Array vArray = null;
        try {
            vArray = axis.read();
        }
        catch (Exception exc) {
            throw new NcAxisException("Could not read axis " + this.name_);
        }
        if (this.numVals_ > 1) {
            this.flipped_ = this.determineFlip();
        }
        if (this.flipped_) {
            vArray = vArray.flip(0);
        }
        for (int j = 0; j < this.numVals_; ++j) {
            double v = vArray.getDouble(vArray.getIndex().set(j));
            if (Double.isNaN(v)) {
                v = 0.0;
            }
            if (j > 0 && v == this.values_[j - 1]) {
                throw new NcAxisException("Axis does not have unique values");
            }
            if (j > 0 && this.type_ == AxisType.Lon) {
                while (v < this.values_[j - 1]) {
                    v += 360.0;
                }
            }
            this.values_[j] = v;
        }
        this.bounds_ = new double[this.numVals_][2];
        boolean gotBounds = this.readBounds();
        if (!gotBounds) {
            gotBounds = this.readEdges();
        }
        if (!gotBounds) {
            gotBounds = this.calculateBounds();
        }
        if (!gotBounds) {
            throw new NcAxisException("Did not get axis value bounds for " + this.name_);
        }
        this.fixBounds();
    }

    private boolean determineFlip() {
        Array vArray;
        try {
            vArray = this.axis_.read();
        }
        catch (Exception exc) {
            return false;
        }
        double val0 = vArray.getDouble(vArray.getIndex().set(0));
        double val1 = vArray.getDouble(vArray.getIndex().set(1));
        if (Double.isNaN(val0)) {
            val0 = 0.0;
        }
        if (Double.isNaN(val1)) {
            val1 = 0.0;
        }
        if (this.type_ == AxisType.Lat) {
            return val0 < val1;
        }
        if (this.type_ == AxisType.Lon) {
            if (this.numVals_ == 2) {
                if (val0 > val1) {
                    return true;
                }
            } else {
                double val2 = vArray.getDouble(vArray.getIndex().set(2));
                if (Double.isNaN(val2)) {
                    val2 = 0.0;
                }
                if (val0 > val1 && val1 > val2) {
                    return true;
                }
            }
        } else {
            if (this.type_ == AxisType.Pressure) {
                return val0 > val1;
            }
            if (this.type_ == AxisType.GeoZ || this.type_ == AxisType.Height) {
                String p = this.axis_.getPositive();
                if (p != null && (p.equals("up") && val1 > val0 || p.equals("down") && val1 < val0)) {
                    return true;
                }
            } else {
                if (this.type_ == AxisType.GeoX) {
                    return val0 > val1;
                }
                if (this.type_ == AxisType.GeoY) {
                    return val0 < val1;
                }
            }
        }
        return false;
    }

    public boolean isPositive() {
        if (this.type_ == AxisType.GeoZ || this.type_ == AxisType.Height) {
            String p = this.axis_.getPositive();
            return p != null && p.equals("up");
        }
        return false;
    }

    private boolean readBounds() {
        int[] bshape;
        Array bArray;
        Attribute ba;
        String boundsName = this.axis_.getBoundaryRef();
        if (boundsName == null && (ba = this.axis_.findAttribute("bounds")) != null) {
            boundsName = ba.getStringValue();
        }
        if (boundsName == null && (ba = this.axis_.findAttribute("climatology")) != null) {
            boundsName = ba.getStringValue();
        }
        if (boundsName == null) {
            return false;
        }
        VariableDS boundsVar = this.dataset_.findVariable(boundsName);
        if (boundsVar == null) {
            return false;
        }
        try {
            bArray = boundsVar.read();
        }
        catch (Exception exc) {
            throw new NcAxisException("Could not read axis bounds " + boundsName);
        }
        if (this.flipped_) {
            bArray = bArray.flip(0);
        }
        if ((bshape = boundsVar.getShape())[0] == 2 && bshape[1] > 2) {
            bArray = bArray.transpose(0, 1);
        }
        if (this.numVals_ > 1) {
            double val0 = this.values_[0];
            double val1 = this.values_[1];
            double b10 = bArray.getDouble(bArray.getIndex().set(1, 0));
            double b11 = bArray.getDouble(bArray.getIndex().set(1, 1));
            if (Double.isNaN(b10)) {
                b10 = 0.0;
            }
            if (Double.isNaN(b11)) {
                b11 = 0.0;
            }
            if (b10 < val0 && b10 < val1 || b10 > val0 && b10 > val1) {
                bArray = bArray.flip(1);
            }
        }
        for (int i = 0; i < this.numVals_; ++i) {
            double b0 = bArray.getDouble(bArray.getIndex().set(i, 0));
            double b1 = bArray.getDouble(bArray.getIndex().set(i, 1));
            if (Double.isNaN(b0)) {
                b0 = 0.0;
            }
            if (Double.isNaN(b1)) {
                b1 = 0.0;
            }
            if (this.type_ == AxisType.Lon) {
                while (b0 > this.values_[i]) {
                    b0 -= 360.0;
                }
                while (b1 < this.values_[i]) {
                    b1 += 360.0;
                }
                while (this.values_[i] - b0 > 360.0) {
                    b0 += 360.0;
                }
                while (b1 - this.values_[i] > 360.0) {
                    b1 -= 360.0;
                }
            }
            this.bounds_[i][0] = b0;
            this.bounds_[i][1] = b1;
        }
        return true;
    }

    private boolean readEdges() {
        Array eArray;
        Attribute edgesA = this.axis_.findAttribute("edges");
        if (edgesA == null) {
            return false;
        }
        String edgesName = edgesA.getStringValue();
        if (edgesName == null) {
            return false;
        }
        VariableDS edgesVar = this.dataset_.findVariable(edgesName);
        try {
            eArray = edgesVar.read();
        }
        catch (Exception exc) {
            throw new NcAxisException("Could not read axis edges " + edgesName);
        }
        if (this.flipped_) {
            eArray = eArray.flip(0);
        }
        for (int i = 0; i < this.numVals_; ++i) {
            double b0 = eArray.getDouble(eArray.getIndex().set(i));
            double b1 = eArray.getDouble(eArray.getIndex().set(i + 1));
            if (Double.isNaN(b0)) {
                b0 = 0.0;
            }
            if (Double.isNaN(b1)) {
                b1 = 0.0;
            }
            if (this.type_ == AxisType.Lon) {
                while (b0 > this.values_[i]) {
                    b0 -= 360.0;
                }
                while (b1 < this.values_[i]) {
                    b1 += 360.0;
                }
                while (this.values_[i] - b0 > 360.0) {
                    b0 += 360.0;
                }
                while (b1 - this.values_[i] > 360.0) {
                    b1 -= 360.0;
                }
            }
            this.bounds_[i][0] = b0;
            this.bounds_[i][1] = b1;
        }
        return true;
    }

    private boolean calculateBounds() {
        if (this.numVals_ > 1) {
            int last = this.numVals_ - 1;
            for (int i = 0; i < last; ++i) {
                this.bounds_[i][1] = 0.5 * (this.values_[i] + this.values_[i + 1]);
                this.bounds_[i + 1][0] = this.bounds_[i][1];
            }
            this.bounds_[0][0] = 2.0 * this.values_[0] - this.bounds_[0][1];
            this.bounds_[last][1] = 2.0 * this.values_[last] - this.bounds_[last][0];
        } else if (this.type_ == AxisType.Lon) {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        } else if (this.type_ == AxisType.Lat) {
            this.bounds_[0][0] = this.values_[0] + 5.0;
            this.bounds_[0][1] = this.values_[0] - 5.0;
        } else {
            this.bounds_[0][0] = this.values_[0] - 5.0;
            this.bounds_[0][1] = this.values_[0] + 5.0;
        }
        return true;
    }

    private void fixBounds() {
        int last = this.numVals_ - 1;
        if (this.type_ == AxisType.Lat) {
            this.bounds_[0][0] = Math.min(90.0, this.bounds_[0][0]);
            this.bounds_[last][1] = Math.max(-90.0, this.bounds_[last][1]);
        } else if (this.type_ == AxisType.Lon) {
            if (this.numVals_ > 2) {
                if (this.values_[last - 1] >= this.values_[0] + 360.0) {
                    this.redundant_ = 2;
                } else if (this.values_[last] >= this.values_[0] + 360.0) {
                    this.redundant_ = 1;
                }
                if (this.redundant_ > 0) {
                    this.numVals_ -= this.redundant_;
                    last = this.numVals_ - 1;
                    double[] newvals = new double[this.numVals_];
                    double[][] newbounds = new double[this.numVals_][2];
                    for (int i = 0; i < this.numVals_; ++i) {
                        newvals[i] = this.values_[i];
                        newbounds[i][0] = this.bounds_[i][0];
                        newbounds[i][1] = this.bounds_[i][1];
                    }
                    this.values_ = newvals;
                    this.bounds_ = newbounds;
                }
            }
            if (this.numVals_ > 1) {
                double delta = this.values_[last] - this.values_[last - 1];
                double overlap = this.bounds_[last][1] - (this.bounds_[0][0] + 360.0);
                boolean bl = this.wraparound_ = overlap >= 0.0 || Math.abs(overlap / delta) < 0.2;
                if (overlap > 0.0) {
                    double[] dArray = this.bounds_[0];
                    dArray[0] = dArray[0] + (overlap *= 0.5);
                    double[] dArray2 = this.bounds_[last];
                    dArray2[1] = dArray2[1] - overlap;
                }
            }
            for (int i = 0; i < this.numVals_; ++i) {
                while (this.bounds_[i][0] > this.values_[i]) {
                    double[] dArray = this.bounds_[i];
                    dArray[0] = dArray[0] - 360.0;
                }
                while (this.values_[i] - this.bounds_[i][0] >= 360.0) {
                    double[] dArray = this.bounds_[i];
                    dArray[0] = dArray[0] + 360.0;
                }
                while (this.bounds_[i][1] < this.values_[i]) {
                    double[] dArray = this.bounds_[i];
                    dArray[1] = dArray[1] + 360.0;
                }
                while (this.bounds_[i][1] - this.values_[i] >= 360.0) {
                    double[] dArray = this.bounds_[i];
                    dArray[1] = dArray[1] - 360.0;
                }
            }
        }
    }

    public CoordinateAxis getAxis() {
        return this.axis_;
    }

    public AxisType getType() {
        return this.type_;
    }

    public String getName() {
        return this.name_;
    }

    public int getSize() {
        return this.values_.length;
    }

    public double[] getValues() {
        return this.values_;
    }

    public double[][] getBounds() {
        return this.bounds_;
    }

    public boolean isFlipped() {
        return this.flipped_;
    }

    public String getLongName() {
        if (this.axis_ != null) {
            return NcVarUtilities.getLongName(this.axis_);
        }
        if (this.type_ == AxisType.GeoX || this.type_ == AxisType.GeoY) {
            return "Axis index";
        }
        return "";
    }

    public String getUnits() {
        if (this.axis_ == null) {
            return "";
        }
        String units = NcVarUtilities.getUnits(this.axis_);
        String lcunits = units.toLowerCase();
        if (lcunits.startsWith("degrees")) {
            if (lcunits.equals("degrees_east")) {
                return "\u00b0E";
            }
            if (lcunits.equals("degrees_north")) {
                return "\u00b0N";
            }
            if (lcunits.equals("degrees_west")) {
                return "\u00b0W";
            }
            if (lcunits.equals("degrees_south")) {
                return "\u00b0S";
            }
        }
        return units;
    }

    public Unit getUcarUnits() {
        String units = this.getUnits();
        if (units == null || units.length() == 0) {
            return null;
        }
        if (unitFormat_ == null) {
            unitFormat_ = UnitFormatManager.instance();
        }
        try {
            return unitFormat_.parse(units);
        }
        catch (Exception exc) {
            return null;
        }
    }

    public double valueAt(int index) {
        return this.values_[index];
    }

    public double[] boundsAt(int index) {
        return new double[]{this.bounds_[index][0], this.bounds_[index][1]};
    }

    public double boundsAt(int index1, int index2) {
        return this.bounds_[index1][index2];
    }

    public int findNearestIndex(double value) {
        if (this.type_ == AxisType.Lat && (value < -90.0 || value > 90.0)) {
            return -1;
        }
        if (this.type_ == AxisType.Lon) {
            int last = this.numVals_ - 1;
            while (value < this.bounds_[0][0]) {
                value += 360.0;
            }
            while (value > this.bounds_[last][1]) {
                value -= 360.0;
            }
        }
        for (int i = 0; i < this.numVals_; ++i) {
            if (value < this.bounds_[i][0] && value < this.bounds_[i][1] || value > this.bounds_[i][0] && value > this.bounds_[i][1]) continue;
            return i;
        }
        return -1;
    }

    public boolean isWraparound() {
        return this.wraparound_;
    }

    public boolean isCompatible(NcAxis pa) {
        if (pa == null) {
            throw new IllegalArgumentException("Null axis.");
        }
        return this.isCompatible(pa.getUcarUnits());
    }

    public boolean isCompatible(Unit u2) {
        Unit u1 = this.getUcarUnits();
        if (u1 == null && u2 == null) {
            return true;
        }
        if (u1 == null || u2 == null) {
            return false;
        }
        return u2.isCompatible(u1);
    }

    public boolean equals(NcAxis a2) {
        if (a2 == this) {
            return true;
        }
        if (a2.getSize() != this.getSize()) {
            return false;
        }
        double[] a2Values = a2.getValues();
        double[][] a2Bounds = a2.getBounds();
        for (int i = 0; i < this.values_.length; ++i) {
            if (a2Values[i] != this.values_[i]) {
                return false;
            }
            if (a2Bounds[i][0] != this.bounds_[i][0]) {
                return false;
            }
            if (a2Bounds[i][1] == this.bounds_[i][1]) continue;
            return false;
        }
        return true;
    }

    protected static boolean isVerticalType(AxisType a) {
        return a == AxisType.Pressure || a == AxisType.Height || a == AxisType.GeoZ;
    }
}

