/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.giss.img;

import gov.nasa.giss.img.ImageUtilities;
import gov.nasa.giss.img.TiffFieldType;
import gov.nasa.giss.img.TiffIfdEntry;
import gov.nasa.giss.img.TiffTag;
import java.awt.image.BufferedImage;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.TreeMap;

public class TiffWriter {
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private BufferedImage image_;
    private String description_;
    private String software_;
    private ArrayList<TiffIfdEntry> ifdEntries_;
    private TreeMap<Integer, AppendedEntry> appendedEntries_;

    public TiffWriter() {
    }

    public TiffWriter(BufferedImage img) {
        this.setImage(img);
    }

    public void flush() {
        this.image_ = null;
        this.description_ = null;
        this.software_ = null;
        this.appendedEntries_ = null;
    }

    public void setImage(BufferedImage img) {
        this.image_ = img;
    }

    public void setDescription(String description) {
        this.description_ = description;
    }

    public void setSoftware(String sw) {
        this.software_ = sw;
    }

    public void write(File f) throws IOException {
        this.write(new FileOutputStream(f));
    }

    public void write(FileOutputStream fos) throws IOException {
        this.write(new DataOutputStream(fos));
    }

    public void write(DataOutputStream dos) throws IOException {
        if (this.image_ == null) {
            throw new RuntimeException("Image has not been specified.");
        }
        TiffWriter.writeHeader(dos);
        int offset = this.writeFirstIfdAndImage(dos, this.appendedEntries_ != null);
        if (offset > 0) {
            offset = this.writeSecondIfdAndData(dos, offset);
        }
        TiffWriter.finish(dos);
    }

    private static void writeHeader(DataOutputStream dos) throws IOException {
        TiffWriter.writeUnsigned1ByteInt(dos, 77);
        TiffWriter.writeUnsigned1ByteInt(dos, 77);
        TiffWriter.writeUnsigned2ByteInt(dos, 42);
        TiffWriter.writeUnsigned4ByteInt(dos, 8);
    }

    private int writeFirstIfdAndImage(DataOutputStream dos, boolean more) throws IOException {
        int offset;
        int numStripBytes;
        String dateString = DATE_FORMAT.format(new Date());
        int iw = this.image_.getWidth();
        int ih = this.image_.getHeight();
        int rowsPerStrip = 200;
        int numStrips = ih / rowsPerStrip;
        int numStripBytesLastStrip = numStripBytes = rowsPerStrip * iw * 3;
        if (numStrips * rowsPerStrip < ih) {
            numStripBytesLastStrip = (ih - numStrips * rowsPerStrip) * iw * 3;
            ++numStrips;
        }
        int numIfdEntries = 16;
        if (this.description_ != null) {
            ++numIfdEntries;
        }
        if (this.software_ != null) {
            ++numIfdEntries;
        }
        int offsetToBps = offset = 10 + numIfdEntries * 12 + 4;
        offset += 8;
        int numDescriptionBytes = 0;
        int offsetToDescription = 0;
        if (this.description_ != null) {
            numDescriptionBytes = this.description_.length() + 1;
            if (numDescriptionBytes % 2 > 0) {
                ++numDescriptionBytes;
            }
            offsetToDescription = offset;
            offset += numDescriptionBytes;
        }
        int offsetToXres = offset;
        int offsetToYres = offset += 8;
        offset += 8;
        int offsetToStripOffsets = 0;
        int offsetToByteCounts = 0;
        if (numStrips > 1) {
            offsetToStripOffsets = offset;
            offsetToByteCounts = offset += numStrips * 4;
            offset += numStrips * 4;
        }
        int numSoftwareBytes = 0;
        int offsetToSoftware = 0;
        if (this.software_ != null) {
            numSoftwareBytes = this.software_.length() + 1;
            if (numSoftwareBytes % 2 > 0) {
                ++numSoftwareBytes;
            }
            offsetToSoftware = offset;
            offset += numSoftwareBytes;
        }
        int numDateBytes = 0;
        int offsetToDate = 0;
        numDateBytes = dateString.length() + 1;
        if (numDateBytes % 2 > 0) {
            ++numDateBytes;
        }
        offsetToDate = offset;
        int offsetToFirstStrip = offset += numDateBytes;
        int offsetToSecondIfd = offset + (numStrips - 1) * numStripBytes + numStripBytesLastStrip;
        TiffWriter.writeUnsigned2ByteInt(dos, numIfdEntries);
        this.writeIfdEntry(dos, TiffTag.NEW_SUBFILE_TYPE, TiffFieldType.LONG, 1, 0);
        this.writeIfdEntry(dos, TiffTag.IMAGE_WIDTH, TiffFieldType.LONG, 1, iw);
        this.writeIfdEntry(dos, TiffTag.IMAGE_LENGTH, TiffFieldType.LONG, 1, ih);
        this.writeIfdEntry(dos, TiffTag.BITS_PER_SAMPLE, TiffFieldType.SHORT, 3, offsetToBps);
        this.writeIfdEntry(dos, TiffTag.COMPRESSION, TiffFieldType.SHORT, 1, 1, 0);
        this.writeIfdEntry(dos, TiffTag.PHOTOMETRIC_INTERPOLATION, TiffFieldType.SHORT, 1, 2, 0);
        if (this.description_ != null) {
            this.writeIfdEntry(dos, TiffTag.IMAGE_DESCRIPTION, TiffFieldType.ASCII, numDescriptionBytes, offsetToDescription);
        }
        if (numStrips == 1) {
            this.writeIfdEntry(dos, TiffTag.STRIP_OFFSETS, TiffFieldType.LONG, numStrips, offsetToFirstStrip);
        } else {
            this.writeIfdEntry(dos, TiffTag.STRIP_OFFSETS, TiffFieldType.LONG, numStrips, offsetToStripOffsets);
        }
        this.writeIfdEntry(dos, TiffTag.ORIENTATION, TiffFieldType.SHORT, 1, 1, 0);
        this.writeIfdEntry(dos, TiffTag.SAMPLES_PER_PIXEL, TiffFieldType.SHORT, 1, 3, 0);
        this.writeIfdEntry(dos, TiffTag.ROWS_PER_STRIP, TiffFieldType.SHORT, 1, rowsPerStrip, 0);
        if (numStrips == 1) {
            this.writeIfdEntry(dos, TiffTag.STRIP_BYTE_COUNT, TiffFieldType.LONG, numStrips, numStripBytes);
        } else {
            this.writeIfdEntry(dos, TiffTag.STRIP_BYTE_COUNT, TiffFieldType.LONG, numStrips, offsetToByteCounts);
        }
        this.writeIfdEntry(dos, TiffTag.X_RESOLUTION, TiffFieldType.RATIONAL, 1, offsetToXres);
        this.writeIfdEntry(dos, TiffTag.Y_RESOLUTION, TiffFieldType.RATIONAL, 1, offsetToYres);
        this.writeIfdEntry(dos, TiffTag.PLANAR_CONFIGURATION, TiffFieldType.SHORT, 1, 1, 0);
        this.writeIfdEntry(dos, TiffTag.RESOLUTION_UNIT, TiffFieldType.SHORT, 1, 2, 0);
        if (offsetToSoftware > 0) {
            this.writeIfdEntry(dos, TiffTag.SOFTWARE, TiffFieldType.ASCII, numSoftwareBytes, offsetToSoftware);
        }
        this.writeIfdEntry(dos, TiffTag.DATE_TIME, TiffFieldType.ASCII, numDateBytes, offsetToDate);
        if (!more) {
            offsetToSecondIfd = 0;
        }
        TiffWriter.writeUnsigned4ByteInt(dos, offsetToSecondIfd);
        dos.flush();
        TiffWriter.writeUnsigned2ByteInt(dos, 8);
        TiffWriter.writeUnsigned2ByteInt(dos, 8);
        TiffWriter.writeUnsigned2ByteInt(dos, 8);
        TiffWriter.writeUnsigned2ByteInt(dos, 0);
        if (offsetToDescription > 0) {
            dos.writeBytes(this.description_);
            dos.writeByte(0);
            if (numDescriptionBytes - this.description_.length() > 1) {
                dos.writeByte(0);
            }
        }
        TiffWriter.writeUnsigned4ByteInt(dos, 1);
        TiffWriter.writeUnsigned4ByteInt(dos, 72);
        TiffWriter.writeUnsigned4ByteInt(dos, 1);
        TiffWriter.writeUnsigned4ByteInt(dos, 72);
        if (numStrips > 1) {
            int i;
            for (i = 0; i < numStrips; ++i) {
                int soffset = offsetToFirstStrip + i * numStripBytes;
                TiffWriter.writeUnsigned4ByteInt(dos, soffset);
            }
            for (i = 0; i < numStrips; ++i) {
                int numBytes = i == numStrips - 1 ? numStripBytesLastStrip : numStripBytes;
                TiffWriter.writeUnsigned4ByteInt(dos, numBytes);
            }
        }
        if (offsetToSoftware > 0) {
            dos.writeBytes(this.software_);
            dos.writeByte(0);
            if (numSoftwareBytes - this.software_.length() > 1) {
                dos.writeByte(0);
            }
        }
        dos.writeBytes(dateString);
        dos.writeByte(0);
        if (numDateBytes - dateString.length() > 1) {
            dos.writeByte(0);
        }
        dos.flush();
        int[] srcPixels = new int[iw * ih];
        try {
            srcPixels = ImageUtilities.getARGBPixels(this.image_);
        }
        catch (Exception exc) {
            exc.printStackTrace();
            throw new RuntimeException("ERROR: Unable to save image as TIFF. See console stack trace.");
        }
        for (int i = 0; i < numStrips; ++i) {
            byte[] stripBytes = i == numStrips - 1 ? new byte[numStripBytesLastStrip] : new byte[numStripBytes];
            int firstRowOfStrip = i * rowsPerStrip;
            int byteOffset = 0;
            for (int row = 0; row < rowsPerStrip && firstRowOfStrip + row < ih; ++row) {
                int srcOffset = (firstRowOfStrip + row) * iw;
                for (int col = 0; col < iw; ++col) {
                    int rgb = srcPixels[srcOffset + col];
                    byte r = (byte)(rgb >> 16 & 0xFF);
                    byte g = (byte)(rgb >> 8 & 0xFF);
                    byte b = (byte)(rgb & 0xFF);
                    stripBytes[byteOffset++] = r;
                    stripBytes[byteOffset++] = g;
                    stripBytes[byteOffset++] = b;
                }
            }
            dos.write(stripBytes, 0, stripBytes.length);
            dos.flush();
        }
        return offsetToSecondIfd;
    }

    void writeIfdEntry(DataOutputStream dos, TiffTag tag, TiffFieldType ftype, int count, int ... vals) throws IOException {
        TiffWriter.writeUnsigned2ByteInt(dos, tag.getCode());
        TiffWriter.writeUnsigned2ByteInt(dos, ftype.getCode());
        TiffWriter.writeUnsigned4ByteInt(dos, count);
        switch (ftype) {
            case SHORT: {
                if (count > 2) {
                    TiffWriter.writeUnsigned4ByteInt(dos, vals[0]);
                    break;
                }
                TiffWriter.writeUnsigned2ByteInt(dos, vals[0]);
                if (count > 1) {
                    TiffWriter.writeUnsigned2ByteInt(dos, vals[1]);
                    break;
                }
                TiffWriter.writeUnsigned2ByteInt(dos, 0);
                break;
            }
            case LONG: {
                TiffWriter.writeUnsigned4ByteInt(dos, vals[0]);
                break;
            }
            case RATIONAL: {
                TiffWriter.writeUnsigned4ByteInt(dos, vals[0]);
                break;
            }
            case ASCII: {
                TiffWriter.writeUnsigned4ByteInt(dos, vals[0]);
            }
        }
    }

    private int writeSecondIfdAndData(DataOutputStream dos, int offset) throws IOException {
        int numValues;
        AppendedEntry entry;
        int i;
        int numIfdEntries = this.appendedEntries_.size();
        int offsetAfterIfd = offset + 2 + numIfdEntries * 12 + 4;
        TiffWriter.writeUnsigned2ByteInt(dos, numIfdEntries);
        Integer[] keys = this.appendedEntries_.keySet().toArray(new Integer[0]);
        int addlOffset = 0;
        for (i = 0; i < numIfdEntries; ++i) {
            entry = this.appendedEntries_.get(keys[i]);
            numValues = entry.getNumValues();
            TiffFieldType ftype = entry.getType();
            int needed = ftype.getSize();
            if (ftype == TiffFieldType.ASCII && numValues % 2 == 1) {
                ++needed;
            }
            if (needed < 4) {
                entry.setOffset(0);
                continue;
            }
            entry.setOffset(offsetAfterIfd + addlOffset);
            addlOffset += needed;
        }
        for (i = 0; i < numIfdEntries; ++i) {
            entry = this.appendedEntries_.get(keys[i]);
            int valOffset = entry.getOffset();
            int numValues2 = entry.getNumValues();
            TiffWriter.writeUnsigned2ByteInt(dos, entry.getTag());
            TiffWriter.writeUnsigned2ByteInt(dos, entry.getType().getCode());
            TiffWriter.writeUnsigned4ByteInt(dos, numValues2);
            if (valOffset != 0) continue;
            Object val = entry.getValues()[0];
        }
        TiffWriter.writeUnsigned4ByteInt(dos, 0);
        for (i = 0; i < numIfdEntries; ++i) {
            entry = this.appendedEntries_.get(keys[i]);
            numValues = entry.getNumValues();
            if (entry.getOffset() <= 0) continue;
            Object[] vals = entry.getValues();
            switch (entry.getType()) {
                // Empty switch
            }
        }
        return 0;
    }

    private static void finish(DataOutputStream dos) throws IOException {
        byte[] padding = new byte[32];
        dos.write(padding, 0, padding.length);
        dos.flush();
        dos.close();
    }

    public void appendEntry(int tag, TiffFieldType type, Object ... values) {
        if (this.appendedEntries_ == null) {
            this.appendedEntries_ = new TreeMap();
        }
        this.appendedEntries_.put(tag, new AppendedEntry(tag, type, values));
    }

    private static void writeUnsigned1ByteInt(DataOutputStream dos, int value) throws IOException {
        dos.writeByte(value);
    }

    private static void writeSigned1ByteInt(DataOutputStream dos, byte value) throws IOException {
        dos.writeByte(value);
    }

    private static void writeUnsigned2ByteInt(DataOutputStream dos, int value) throws IOException {
        dos.writeShort(value);
    }

    private static void writeSigned2ByteInt(DataOutputStream dos, short value) throws IOException {
        dos.writeShort(value);
    }

    private static void writeUnsigned4ByteInt(DataOutputStream dos, int value) throws IOException {
        dos.writeInt(value);
    }

    private static void writeSigned4ByteInt(DataOutputStream dos, int value) throws IOException {
        dos.writeInt(value);
    }

    class AppendedEntry {
        private int tag_;
        private TiffFieldType type_;
        private Object[] values_;
        private int offset_;

        AppendedEntry(int tag, TiffFieldType type, Object ... values) {
            this.tag_ = tag;
            this.type_ = type;
            this.values_ = values;
        }

        int getTag() {
            return this.tag_;
        }

        TiffFieldType getType() {
            return this.type_;
        }

        int getNumValues() {
            return this.values_.length;
        }

        Object[] getValues() {
            return this.values_;
        }

        int getOffset() {
            return this.offset_;
        }

        void setOffset(int offset) {
            this.offset_ = offset;
        }
    }
}

